<?php
/**
 * Mailchimp Subscribe Widget
 *
 * There are two hidden inputs, where the account ID and the selected list ID are saved.
 * These two fields get populated via JS, because we get the email lists from the MC API.
 *
 * @package pt-mcw
 */
 
add_action('widgets_init','geo_subscribe_form');

function geo_subscribe_form(){
	register_widget("PT_Mailchimp_Subscribe");
}

if ( ! class_exists( 'PT_Mailchimp_Subscribe' ) ) {
	class PT_Mailchimp_Subscribe extends WP_Widget {
		/**
		 * Register widget with WordPress.
		 */
		public function __construct() {
			parent::__construct(
				false, // ID, auto generate when false.
				THEME_NAME .' - '.esc_html__( 'E-Posta Aboneliği', 'geo' ),
				array(
					'classname'   => 'widget-mailchimp-subscribe',
				)
			);
			// AJAX callback.
			add_action( 'wp_ajax_pt_mailchimp_subscribe_get_lists', array( $this, 'mailchimp_get_lists' ) );
		}
		/**
		 * Front-end display of widget.
		 *
		 * @see WP_Widget::widget()
		 *
		 * @param array $args
		 * @param array $instance
		 */
		public function widget( $args, $instance ) {
			$api_key        = empty( $instance['api_key'] ) ? '' : $instance['api_key'];
			$account_id     = empty( $instance['account_id'] ) ? '' : $instance['account_id'];
			$selected_list  = empty( $instance['selected_list'] ) ? '' : $instance['selected_list'];
			$subscribe_text = empty( $instance['subscribe_text'] ) ? '' : $instance['subscribe_text'];
			$title 			= empty( $instance['title'] ) ? '' : $instance['title'];
			$mc_datacenter  = '';
			$form_action    = '';
			if ( ! empty( $api_key ) && preg_match( '/us\d{1,2}$/', $api_key, $mc_dc_match ) ) {
				$mc_datacenter = $mc_dc_match[0];
				$form_action = sprintf(
					'//github.%1$s.list-manage.com/subscribe/post?u=%2$s&amp;id=%3$s',
					esc_attr( $mc_datacenter ),
					esc_attr( $account_id ),
					esc_attr( $selected_list )
				);
			}
			$mc_securty_string = sprintf( 'b_%1$s_%2$s', esc_attr( $account_id ), esc_attr( $selected_list ) );
			$form_texts = apply_filters( 'pt-mcw/form_texts', array(
				'email'  => esc_html__( 'E-Posta Adresiniz', 'geo' ),
				'submit' => esc_html__( 'Abone Ol', 'geo' ),
			) );
			echo $args['before_widget'];
			?>
			<div class="mailchimp_subsribe">
				<?php if(!empty($title)): ?>
					<h4><?php echo $title; ?></h4>
				<?php endif; ?>
				<?php if(!empty($subscribe_text)): ?>
					<p><?php echo $subscribe_text; ?></p>
				<?php endif; ?>
				<form <?php echo ! empty( $form_action ) ? 'action="' . esc_url( $form_action ) . '"' : ''; ?> method="post" name="mc-embedded-subscribe-form" class="mailchimp-subscribe  validate" target="_blank" novalidate>
					<div class="form-group">
						<div class="form-label-group">
							<input type="email" value="" name="EMAIL" id="EMAIL" class="email form-control mailchimp-subscribe__email-input" placeholder="<?php echo esc_html( $form_texts['email'] ); ?>" required>
							<label for="EMAIL"><?php _e('E-Posta Adresiniz', 'geo'); ?></label>
						</div>
					</div>
					<div style="position: absolute; left: -5000px;" aria-hidden="true"><input type="text" name="<?php echo esc_attr( $mc_securty_string ); ?>" tabindex="-1" value=""></div>
					
					<input type="submit" value="<?php echo esc_html( $form_texts['submit'] ); ?>" name="subscribe" class="button  btn  btn-primary mailchimp-subscribe__submit">
				</form>
			</div>
			<?php
			echo $args['after_widget'];
		}
		/**
		 * Sanitize widget form values as they are saved.
		 *
		 * @param array $new_instance The new options.
		 * @param array $old_instance The previous options.
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = array();
			$instance['api_key']       = sanitize_text_field( $new_instance['api_key'] );
			$instance['account_id']    = sanitize_text_field( $new_instance['account_id'] );
			$instance['selected_list'] = sanitize_text_field( $new_instance['selected_list'] );
			$instance['subscribe_text'] = sanitize_text_field( $new_instance['subscribe_text'] );
			$instance['title'] = sanitize_text_field( $new_instance['title'] );
			return $instance;
		}
		/**
		 * Back-end widget form.
		 *
		 * @param array $instance The widget options.
		 */
		public function form( $instance ) {
			$api_key       = empty( $instance['api_key'] ) ? '' : $instance['api_key'];
			$account_id    = empty( $instance['account_id'] ) ? '' : $instance['account_id'];
			$selected_list = empty( $instance['selected_list'] ) ? '' : $instance['selected_list'];
			$defaults = array('title' => __('Bültenimize Katılın', 'geo'), 'subscribe_text' => __('Hemen ücretsiz üye olun ve yeni güncellemelerden haberdar olan ilk kişi olun.', 'geo'));
			$instance = wp_parse_args((array) $instance, $defaults); ?>
			
			<p>			
				<label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e('Başlık', 'geo') ?></label>
				<input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" value="<?php echo $instance['title']; ?>" />
			</p>
			<p>
				<?php esc_html_e( 'Bu bileşeni kullanabilmek için şunları yapmanız gerekir: ', 'geo' ); ?>
			</p>

			<ol>
				<li><?php printf( esc_html__( '%1$sBu URL\'yi ziyaret edin ve mailchimp hesabınızla giriş yapın.%2$s,', 'geo' ), '<a href="https://admin.mailchimp.com/account/api" target="_blank">', '</a>' ); ?></li>
				<li><?php printf( esc_html__( '%1$sAPI anahtarı oluşturun%2$s ve aşağıdaki giriş alanına yapıştırın,', 'geo' ), '<a href="http://kb.mailchimp.com/integrations/api-integrations/about-api-keys#Find-or-Generate-Your-API-Key" target="_blank">', '</a>' ); ?></li>
				<li><?php esc_html_e( 'Mevcut MailChimp listelerinizin alınabilmesi için Bağlan düğmesine tıklayın,', 'geo' ); ?></li>
				<li><?php esc_html_e( 'Aşağıdaki açılır listeden ziyaretçilerinizin abone olmasını istediğiniz listeyi seçin.', 'geo' ); ?></li>
			</ol>

			<p>
				<label for="<?php echo $this->get_field_id( 'api_key' ); ?>"><?php esc_html_e( 'MailChimp API key:', 'geo' ); ?>
				</label>
				<input class="js-mailchimp-api-key" id="<?php echo $this->get_field_id( 'api_key' ); ?>" name="<?php echo $this->get_field_name( 'api_key' ); ?>" type="text" value="<?php echo esc_html( $api_key ); ?>" />
				<input class="js-connect-mailchimp-api-key  button" type="button" value="<?php esc_html_e( 'Bağlan', 'geo' ); ?>">
				<input class="js-mailchimp-account-id" id="<?php echo $this->get_field_id( 'account_id' ); ?>" name="<?php echo $this->get_field_name( 'account_id' ); ?>" type="hidden" value="<?php echo esc_attr( $account_id ); ?>">
			</p>

			<p class="js-mailchimp-loader" style="display: none;">
				<span class="spinner" style="display: inline-block; float: none; visibility: visible; margin-bottom: 6px;" ></span> <?php esc_html_e( 'Loading ...', 'geo' ); ?>
			</p>

			<div class="js-mailchimp-notice"></div>

			<p class="js-mailchimp-list-container" style="display: none;">
				<label for="<?php echo $this->get_field_id( 'list' ); ?>"><?php esc_html_e( 'MailChimp list:', 'geo' ); ?></label> <br>
				<select id="<?php echo $this->get_field_id( 'list' ); ?>" name="<?php echo $this->get_field_name( 'list' ); ?>"></select>
				<input class="js-mailchimp-selected-list" id="<?php echo $this->get_field_id( 'selected_list' ); ?>" name="<?php echo $this->get_field_name( 'selected_list' ); ?>" type="hidden" value="<?php echo esc_attr( $selected_list ); ?>">
			</p>

			<?php if ( ! empty( $api_key ) ) : ?>
				<script type="text/javascript">
					jQuery( '.js-connect-mailchimp-api-key' ).trigger( 'click' );
				</script>
			<?php endif; ?>
			
			<p>
			<label for="<?php echo $this->get_field_id( 'subscribe_text' ); ?>"><?php _e('Açıklama', 'geo'); ?></label>
			<textarea style="height:200px;" class="widefat" id="<?php echo $this->get_field_id( 'subscribe_text' ); ?>" name="<?php echo $this->get_field_name( 'subscribe_text' ); ?>"><?php echo stripslashes(htmlspecialchars(( $instance['subscribe_text'] ), ENT_QUOTES)); ?></textarea>
     </p>

			<?php
		}
		/**
		 * AJAX callback function to retrieve the MailChimp lists.
		 */
		public function mailchimp_get_lists() {
			check_ajax_referer( 'pt-mcw-ajax-verification', 'security' );
			$response = array();
			$api_key       = sanitize_text_field( $_GET['api_key'] );
			$mc_datacenter = sanitize_text_field( $_GET['mc_dc'] );
			$args = array(
				'headers' => array(
					'Authorization' => sprintf( 'apikey %1$s', $api_key ),
				),
			);
			$mc_lists_endpoint = sprintf( 'https://%1$s.api.mailchimp.com/3.0/lists', $mc_datacenter );
			$request = wp_remote_get( $mc_lists_endpoint, $args );
			// Error while connecting to the MailChimp server.
			if ( is_wp_error( $request ) ) {
				$response['message'] = esc_html__( 'There was an error connecting to the MailChimp servers.', 'pt-mcw' );
				wp_send_json_error( $response );
			}
			// Retrieve the response code and body.
			$response_code = wp_remote_retrieve_response_code( $request );
			$response_body = json_decode( wp_remote_retrieve_body( $request ), true );
			// The request was not successful.
			if ( 200 !== $response_code ) {
				$response['message'] = sprintf( esc_html__( 'Error: %1$s (error code: %2$s)', 'pt-mcw' ), $response_body['title'], $response_body['status'] );
				wp_send_json_error( $response );
			}
			// There are no lists in this MailChimp account.
			if ( empty( $response_body['lists'] ) ) {
				$response['message'] = esc_html__( 'There are no email lists with this API key! Please create an email list in the MailChimp dashboard and try again.', 'pt-mcw' );
				wp_send_json_error( $response );
			}
			$mc_account_id = $this->get_mailchimp_account_id( $api_key, $mc_datacenter );
			if ( empty( $mc_account_id ) ) {
				$response['message'] = esc_html__( 'There was an error connecting to the MailChimp servers.', 'pt-mcw' );
				wp_send_json_error( $response );
			}
			$lists = array();
			// Parse through the retrieved lists and collect the info we need.
			foreach ( $response_body['lists'] as $list ) {
				$lists[ $list['id'] ] = $list['name'];
			}
			$response['message']    = esc_html__( 'MailChimp lists were successfully retrieved!', 'pt-mcw' );
			$response['lists']      = $lists;
			$response['account_id'] = $mc_account_id;
			wp_send_json_success( $response );
		}
		/**
		 * Get the mailchimp account ID from the API key.
		 *
		 * @return string API key
		 */
		private function get_mailchimp_account_id( $api_key, $mc_datacenter ) {
			$args = array(
				'headers' => array(
					'Authorization' => sprintf( 'apikey %1$s', $api_key ),
				),
			);
			$mc_account_endpoint = sprintf( 'https://%1$s.api.mailchimp.com/3.0/', $mc_datacenter );
			$request = wp_remote_get( $mc_account_endpoint, $args );
			if ( is_wp_error( $request ) || 200 !== wp_remote_retrieve_response_code( $request ) ) {
				return false;
			}
			$body = json_decode( wp_remote_retrieve_body( $request ), true );
			return $body['account_id'];
		}
	}
}