<?php
/**
 * Theme Update Notifier
 *
 * Provides a notification to the user everytime the WordPress theme is updated.
 *
 * @package     WebMan WordPress Theme Framework
 * @subpackage  Theme Update Notifier
 * @author      Modifications by WebMan - Oliver Juhas
 * @author      Joao Araujo
 * @link        http://themeforest.net/user/unisphere
 * @link        http://twitter.com/unispheredesign
 *
 * @since    3.0
 * @version  1.6
 *
 * CONTENT:
 * - 1) Constants
 * - 10) Actions and filters
 * - 20) Dashboard menu and admin bar
 * - 30) Update notifier page
 * - 40) Remote XML data
 *****************************************************
 */

/**
 * 1) Constants
 */

	//The remote notifier XML file containing the latest version of the theme and changelog
		define( 'NOTIFIER_XML_FILE', 'https://kanthemesdev.github.io/geoit-update/update.xml' );
	//The time interval for the remote XML cache in the database (86400 seconds = 24 hours)
		define( 'NOTIFIER_CACHE_INTERVAL', 86400 );
		define( 'THEME_SHORTNAME', 'geo' );



if( is_admin() ):

/**
 * 10) Actions and filters
 */

	/**
	 * Actions
	 */

		//Admin menu and menu bar
			if ( ! apply_filters( 'wmhook_disable_update_notifier_bar_menu', false ) ) {
				add_action( 'admin_bar_menu', 'update_notifier_bar_menu', 1000 );
			}



	/**
	 * Adds an update notification to the WordPress 3.1+ Admin Bar
	 */
	function update_notifier_bar_menu() {
		if ( function_exists( 'simplexml_load_string' ) ) { //Stop if simplexml_load_string funtion isn't available
			if ( ! is_super_admin() || ! is_admin_bar_showing() ) {
			//Don't display notification in admin bar if it's disabled or the current user isn't an administrator
				return;
			}

			global $wp_admin_bar;

			$xml = get_latest_theme_version( NOTIFIER_CACHE_INTERVAL ); //Get the latest remote XML file on our server

			if (
					isset( $xml->latest )
					&& version_compare( $xml->latest, THEME_VER, '>' )
				) {

				$adminURL = get_admin_url() . 'admin.php?page=geoit-update';
				$wp_admin_bar->add_menu( array(
						'id'    => 'update_notifier',
						'title' => sprintf( __( '%s Güncellemesi Hazır', 'geoit' ), THEME_NAME ) . ' <span id="ab-updates">1</span>',
						'href'  => $adminURL
					) );

			}

		}
	} // /update_notifier_bar_menu


/**
 * 40) Remote XML data
 */

	/**
	 * Remove XML file processing
	 *
	 * Get the remote XML file contents and return its data (Version and Changelog).
	 * Uses the cached version if available and inside the time interval defined.
	 *
	 * @param  integer $interval
	 */
	function get_latest_theme_version( $interval ) {
		//Requirements check
			if ( ! is_super_admin() ) {
				return;
			}

		$db_cache_field              = 'notifier-cache-' . THEME_SHORTNAME;
		$db_cache_field_last_updated = 'notifier-cache-' . THEME_SHORTNAME . '-last-updated';
		$last                        = get_option( $db_cache_field_last_updated );
		$now                         = time();
		$interval                    = absint( $interval );

		//check the cache
		if (
				! $last
				|| ( time() - $last ) > absint( $interval )
				|| 3 > absint( get_option( THEME_SHORTNAME.'update' ) )
			) {

			//cache doesn't exist, or is old, so refresh it
			$response = wp_remote_get( NOTIFIER_XML_FILE );

			if ( is_wp_error( $response ) ) {

				$error = $response->get_error_message();

				$cache  = '<?xml version="1.0" encoding="UTF-8"?>';
				$cache .= '<notifier>';
					$cache .= '<latest>1.0</latest>';
					$cache .= '<message><![CDATA[<span style="font-size:125%;color:#f33">Something went wrong: ' . wp_kses(
							$error,
							array(
								'a' => array(
										'href' => true,
										'class' => true,
									),
								'span' => array(
										'class' => true,
									),
								'strong' => array(
										'class' => true,
									),
							)
						) . '</span>]]></message>';
					$cache .= '<changelog></changelog>';
					$cache .= '<changefiles></changefiles>';
				$cache .= '</notifier>';

			} else {

				$cache = $response['body'];

			}

			if ( $cache ) {
				//we got good results
				update_option( $db_cache_field, $cache );
				update_option( $db_cache_field_last_updated, time() );
			}

			//read from the cache file
			$notifier_data = get_option( $db_cache_field );

		} else {

			//cache file is fresh enough, so read from it
			$notifier_data = get_option( $db_cache_field );

		}

		//Let's see if the $xml data was returned as we expected it to.
		//If it didn't, use the default 1.0 as the latest version so that we don't have problems when the remote server hosting the XML file is down
		if ( strpos( (string) $notifier_data, '<notifier>' ) === false ) {

			$notifier_data  = '<?xml version="1.0" encoding="UTF-8"?>';
			$notifier_data .= '<notifier>';
				$notifier_data .= '<latest>1.0</latest>';
				$notifier_data .= '<message></message>';
				$notifier_data .= '<changelog></changelog>';
				$notifier_data .= '<changefiles></changefiles>';
			$notifier_data .= '</notifier>';

		}

		//Load the remote XML data into a variable and return it
		$xml = simplexml_load_string( $notifier_data );

		return $xml;
	} // /get_latest_theme_version
	
endif;